/*
 * dnscrypt.c
 *
 * Copyright (C) 2020 - ntop.org
 *
 * nDPI is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * nDPI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with nDPI.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "ndpi_protocol_ids.h"

#define NDPI_CURRENT_PROTO NDPI_PROTOCOL_DNSCRYPT

#include "ndpi_api.h"
#include "ndpi_private.h"

static void ndpi_int_dnscrypt_add_connection(struct ndpi_detection_module_struct *ndpi_struct,
                                             struct ndpi_flow_struct *flow)
{
  ndpi_set_detected_protocol(ndpi_struct, flow, NDPI_PROTOCOL_DNSCRYPT, NDPI_PROTOCOL_UNKNOWN, NDPI_CONFIDENCE_DPI);
}

static void ndpi_search_dnscrypt(struct ndpi_detection_module_struct *ndpi_struct,
                                 struct ndpi_flow_struct *flow)
{
  struct ndpi_packet_struct *packet = &ndpi_struct->packet;
  static char const * const dnscrypt_initial = "2\rdnscrypt";

  NDPI_LOG_DBG(ndpi_struct, "search dnscrypt\n");

  /* dnscrypt protocol version 1: check magic */
  if (packet->payload_packet_len >= 64 &&
      strncmp((char*)packet->payload, "r6fnvWj8", strlen("r6fnvWj8")) == 0)
  {
    ndpi_int_dnscrypt_add_connection(ndpi_struct, flow);
    return;
  }
  
  /* dnscrypt protocol version 1 and 2: resolver ping */
  if (packet->payload_packet_len > 13 + strlen(dnscrypt_initial) &&
      strncasecmp((char*)packet->payload + 13, dnscrypt_initial, strlen(dnscrypt_initial)) == 0)
  {
    ndpi_int_dnscrypt_add_connection(ndpi_struct, flow);
    return;
  }

  if ((flow->packet_direction_counter[packet->packet_direction] >= 1 &&
       flow->packet_direction_counter[1 - packet->packet_direction] >= 1) ||
      flow->packet_counter >= 8) {
    /*
     * Wait for at least one packet per direction, up to a max
     * Required as we need to wait for the server response which contains the ASCII pattern below.
     */
    NDPI_EXCLUDE_PROTO(ndpi_struct, flow);
    return;
  }
}

void init_dnscrypt_dissector(struct ndpi_detection_module_struct *ndpi_struct, u_int32_t *id)
{
  ndpi_set_bitmask_protocol_detection(
    "DNScrypt", ndpi_struct, *id,
    NDPI_PROTOCOL_DNSCRYPT, ndpi_search_dnscrypt, NDPI_SELECTION_BITMASK_PROTOCOL_V4_V6_TCP_OR_UDP_WITH_PAYLOAD_WITHOUT_RETRANSMISSION,
    SAVE_DETECTION_BITMASK_AS_UNKNOWN, ADD_TO_DETECTION_BITMASK);
  *id += 1;
}

