#ifdef HAVE_PCRE2

#ifndef NDPI_RCE_H
#define NDPI_RCE_H

#endif //NDPI_RCE_H

#define N_RCE_REGEX 7

/* Compiled regex */
static struct pcre2_struct *comp_rx[N_RCE_REGEX];

static unsigned int initialized_comp_rx = 0;

static const char *rce_regex[N_RCE_REGEX] = {
/**
 * https://github.com/SpiderLabs/owasp-modsecurity-crs/blob/v3.3/dev/rules/REQUEST-932-APPLICATION-ATTACK-RCE.conf
 */

/**
 * [ Unix command injection ]
 *
 * This regex detects Unix command injections.
 * A command injection takes a form such as:
 *
 *   foo.jpg;uname -a
 *   foo.jpg||uname -a
 *
 * The vulnerability exists when an application executes a shell command
 * without proper input escaping/validation.
 *
 * This regex is also triggered by an Oracle WebLogic Remote Command Execution exploit: [ Oracle WebLogic vulnerability CVE-2017-10271 - Exploit tested: https://www.exploit-db.com/exploits/43458 ]
 *
 * To prevent false positives, we look for a 'starting sequence' that
 * precedes a command in shell syntax, such as: ; | & $( ` <( >(
 * Anatomy of the regexp with examples of patterns caught:
 *
 * 1. Starting tokens
 *
 * ;        ;ifconfig
 * \{       {ifconfig}
 * \|       |ifconfig
 * \|\|     ||ifconfig
 * &        &ifconfig
 * &&       &&ifconfig
 * \n       ;\nifconfig
 * \r       ;\rifconfig
 * \$\(     $(ifconfig)
 * $\(\(    $((ifconfig))
 * `        `ifconfig`
 * \${      ${ifconfig}
 * <\(      <( ifconfig )
 * >\(      >( ifconfig )
 * \(\s*\)  a() ( ifconfig; ); a
 *
 * 2. Command prefixes
 *
 * {        { ifconfig }
 * \s*\(\s* ( ifconfig )
 * \w+=(?:[^\s]*|\$.*|\$.*|<.*|>.*|\'.*\'|\".*\")\s+   VARNAME=xyz ifconfig
 * !\s*     ! ifconfig
 * \$       $ifconfig
 *
 * 3. Quoting
 *
 * '        'ifconfig'
 * \"       "ifconfig"
 *
 * 4. Paths
 *
 * [\?\*\[\]\(\)\-\|+\w'\"\./\\\\]+/   /sbin/ifconfig, /s?in/./ifconfig, /s[a-b]in/ifconfig etc.
 *
 * This regex is case-sensitive to prevent FP ("Cat" vs. "cat").
 *
 * An effort was made to combat evasions by shell quoting (e.g. 'ls',
 * 'l'"s", \l\s are all valid).
 *
 * This is the base regex to prevent Unix Command Injection
 */

  "(?:;|\\{|\\||\\|\\||&|&&|\\n|\\r|\\$\\(|\\$\\(\\(|`|\\${|<\\(|>\\(|\\(\\s*\\))\\s*(?:{|\\s*\\(\\s*|\\w+=(?:[^\\s]*|\\$.*|\\$.*|<.*|>.*|\\'.*\\'|\\\".*\\\")\\s+|!\\s*|\\$)*\\s*(?:'|\\\")*(?:[\\?\\*\\[\\]\\(\\)\\-\\|+\\w'\\\"\\./\\\\\\\\]+/)?[\\\\\\\\'\\\"]*(?:l[\\\\\\\\'\\\"]*(?:w[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*-[\\\\\\\\'\\\"]*(?:d[\\\\\\\\'\\\"]*(?:o[\\\\\\\\'\\\"]*w[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*d|u[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*p)|r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*q[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*t|m[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*r)|s(?:[\\\\\\\\'\\\"]*(?:b[\\\\\\\\'\\\"]*_[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*e|c[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*u|m[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*d|p[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*i|u[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*b|-[\\\\\\\\'\\\"]*F|h[\\\\\\\\'\\\"]*w|o[\\\\\\\\'\\\"]*f))?|z[\\\\\\\\'\\\"]*(?:(?:[ef][\\\\\\\\'\\\"]*)?g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p|c[\\\\\\\\'\\\"]*(?:a[\\\\\\\\'\\\"]*t|m[\\\\\\\\'\\\"]*p)|m[\\\\\\\\'\\\"]*(?:o[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e|a)|d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s)|e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*(?:(?:f[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*l|p[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*p)[\\\\\\\\'\\\"]*e|e[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*o|(?:\\s|<|>).*)|a[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*(?:l[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*g(?:[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n)?|c[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*m|(?:\\s|<|>).*)|o[\\\\\\\\'\\\"]*(?:c[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*(?:t[\\\\\\\\'\\\"]*e|l)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|g[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*e)|d[\\\\\\\\'\\\"]*(?:c[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*g|d[\\\\\\\\'\\\"]*(?:\\s|<|>).*)|f[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*p(?:[\\\\\\\\'\\\"]*g[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*t)?|(?:[np]|y[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*x)[\\\\\\\\'\\\"]*(?:\\s|<|>).*)|b[\\\\\\\\'\\\"]*(?:z[\\\\\\\\'\\\"]*(?:(?:[ef][\\\\\\\\'\\\"]*)?g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p|d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s|m[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e|c[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t|i[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*2)|s[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*(?:c[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t|i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|t[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*r)|a[\\\\\\\\'\\\"]*(?:t[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*(?:\\s|<|>).*|s[\\\\\\\\'\\\"]*h)|r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*k[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*w|u[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n)|c[\\\\\\\\'\\\"]*(?:o[\\\\\\\\'\\\"]*(?:m[\\\\\\\\'\\\"]*(?:p[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s|m[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*d)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|p[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*c)|h[\\\\\\\\'\\\"]*(?:d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*(?:\\s|<|>).*|f[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*g[\\\\\\\\'\\\"]*s|a[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*r|m[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*d)|r[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*b|(?:[cp]|a[\\\\\\\\'\\\"]*t)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|u[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*l|s[\\\\\\\\'\\\"]*h)|f[\\\\\\\\'\\\"]*(?:i(?:[\\\\\\\\'\\\"]*(?:l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*(?:t[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*t|(?:\\s|<|>).*)|n[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*(?:\\s|<|>).*))?|t[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*(?:s[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*s|w[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*o|(?:\\s|<|>).*)|u[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*n|(?:e[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*h|c)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|o[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*h|g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p)|e[\\\\\\\\'\\\"]*(?:n[\\\\\\\\'\\\"]*(?:v(?:[\\\\\\\\'\\\"]*-[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*e)?|d[\\\\\\\\'\\\"]*(?:i[\\\\\\\\'\\\"]*f|s[\\\\\\\\'\\\"]*w))|x[\\\\\\\\'\\\"]*(?:p[\\\\\\\\'\\\"]*(?:a[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*d|o[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*t|r)|e[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*(?:\\s|<|>).*)|c[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*(?:\\s|<|>).*|g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p|s[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*c|v[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*l)|h[\\\\\\\\'\\\"]*(?:t[\\\\\\\\'\\\"]*(?:d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*g[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*t|p[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*w[\\\\\\\\'\\\"]*d)|o[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*(?:n[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*e|i[\\\\\\\\'\\\"]*d)|(?:e[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*d|u[\\\\\\\\'\\\"]*p)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|i[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*y)|i[\\\\\\\\'\\\"]*(?:p[\\\\\\\\'\\\"]*(?:(?:6[\\\\\\\\'\\\"]*)?t[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*b[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s|c[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*g)|r[\\\\\\\\'\\\"]*b(?:[\\\\\\\\'\\\"]*(?:1(?:[\\\\\\\\'\\\"]*[89])?|2[\\\\\\\\'\\\"]*[012]))?|f[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*g|d[\\\\\\\\'\\\"]*(?:\\s|<|>).*)|g[\\\\\\\\'\\\"]*(?:(?:e[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*l|r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p|c[\\\\\\\\'\\\"]*c|i[\\\\\\\\'\\\"]*t)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|z[\\\\\\\\'\\\"]*(?:c[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t|i[\\\\\\\\'\\\"]*p)|u[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*z[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*p|d[\\\\\\\\'\\\"]*b)|a[\\\\\\\\'\\\"]*(?:(?:l[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*s|w[\\\\\\\\'\\\"]*k)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|d[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*r|p[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*-[\\\\\\\\'\\\"]*g[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*t|r[\\\\\\\\'\\\"]*(?:c[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*(?:\\s|<|>).*|p))|d[\\\\\\\\'\\\"]*(?:h[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*t|(?:i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|u)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|(?:m[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s|p[\\\\\\\\'\\\"]*k)[\\\\\\\\'\\\"]*g|o[\\\\\\\\'\\\"]*(?:a[\\\\\\\\'\\\"]*s|n[\\\\\\\\'\\\"]*e)|a[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*h)|m[\\\\\\\\'\\\"]*(?:(?:k[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*r|o[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|a[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*(?:x[\\\\\\\\'\\\"]*(?:\\s|<|>).*|q)|l[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*e)|j[\\\\\\\\'\\\"]*(?:(?:a[\\\\\\\\'\\\"]*v[\\\\\\\\'\\\"]*a|o[\\\\\\\\'\\\"]*b[\\\\\\\\'\\\"]*s)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|e[\\\\\\\\'\\\"]*x[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*c)|k[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*(?:a[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*l|(?:\\s|<|>).*)|(?:G[\\\\\\\\'\\\"]*E[\\\\\\\\'\\\"]*T[\\\\\\\\'\\\"]*(?:\\s|<|>)|\\.\\s).*|7[\\\\\\\\'\\\"]*z(?:[\\\\\\\\'\\\"]*[ar])?)\\b",

  "(?:;|\\{|\\||\\|\\||&|&&|\\n|\\r|\\$\\(|\\$\\(\\(|`|\\${|<\\(|>\\(|\\(\\s*\\))\\s*(?:{|\\s*\\(\\s*|\\w+=(?:[^\\s]*|\\$.*|\\$.*|<.*|>.*|\\'.*\\'|\\\".*\\\")\\s+|!\\s*|\\$)*\\s*(?:'|\\\")*(?:[\\?\\*\\[\\]\\(\\)\\-\\|+\\w'\\\"\\./\\\\\\\\]+/)?[\\\\\\\\'\\\"]*(?:s[\\\\\\\\'\\\"]*(?:e[\\\\\\\\'\\\"]*(?:t[\\\\\\\\'\\\"]*(?:(?:f[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*)?(?:\\s|<|>).*|e[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*v|s[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*d)|n[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*l|d[\\\\\\\\'\\\"]*(?:\\s|<|>).*)|h[\\\\\\\\'\\\"]*(?:\\.[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*b|u[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*w[\\\\\\\\'\\\"]*n|(?:\\s|<|>).*)|o[\\\\\\\\'\\\"]*(?:(?:u[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*e|r[\\\\\\\\'\\\"]*t)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|c[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t)|c[\\\\\\\\'\\\"]*(?:h[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*d|p[\\\\\\\\'\\\"]*(?:\\s|<|>).*)|t[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*g[\\\\\\\\'\\\"]*s|(?:l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*e|f[\\\\\\\\'\\\"]*t)[\\\\\\\\'\\\"]*p|y[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*l|u[\\\\\\\\'\\\"]*(?:(?:\\s|<|>).*|d[\\\\\\\\'\\\"]*o)|d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|s[\\\\\\\\'\\\"]*h|v[\\\\\\\\'\\\"]*n)|p[\\\\\\\\'\\\"]*(?:k[\\\\\\\\'\\\"]*(?:g(?:(?:[\\\\\\\\'\\\"]*_)?[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*o)?|e[\\\\\\\\'\\\"]*x[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*c|i[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*l)|t[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*r(?:[\\\\\\\\'\\\"]*(?:d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p))?|a[\\\\\\\\'\\\"]*(?:t[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*(?:\\s|<|>).*|s[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*w[\\\\\\\\'\\\"]*d)|r[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*(?:e[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*v|f[\\\\\\\\'\\\"]*(?:\\s|<|>).*)|y[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*n(?:[\\\\\\\\'\\\"]*(?:3(?:[\\\\\\\\'\\\"]*m)?|2))?|e[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*(?:l(?:[\\\\\\\\'\\\"]*(?:s[\\\\\\\\'\\\"]*h|5))?|m[\\\\\\\\'\\\"]*s)|(?:g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e|f[\\\\\\\\'\\\"]*t)[\\\\\\\\'\\\"]*p|(?:u[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*h|o[\\\\\\\\'\\\"]*p)[\\\\\\\\'\\\"]*d|h[\\\\\\\\'\\\"]*p(?:[\\\\\\\\'\\\"]*[57])?|i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*g|s[\\\\\\\\'\\\"]*(?:\\s|<|>).*)|n[\\\\\\\\'\\\"]*(?:c[\\\\\\\\'\\\"]*(?:\\.[\\\\\\\\'\\\"]*(?:t[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*l|o[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*b[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*d)|(?:\\s|<|>).*|a[\\\\\\\\'\\\"]*t)|e[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*(?:k[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*-[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*p|(?:s[\\\\\\\\'\\\"]*t|c)[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t|(?:\\s|<|>).*)|s[\\\\\\\\'\\\"]*(?:l[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*k[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*p|t[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t)|(?:a[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*o|i[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*e)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|(?:o[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*u|m[\\\\\\\\'\\\"]*a)[\\\\\\\\'\\\"]*p|p[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*g)|r[\\\\\\\\'\\\"]*(?:e[\\\\\\\\'\\\"]*(?:(?:p[\\\\\\\\'\\\"]*(?:l[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*e|e[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t)|n[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*e)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|a[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*h)|m[\\\\\\\\'\\\"]*(?:(?:d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*)?(?:\\s|<|>).*|u[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*r)|u[\\\\\\\\'\\\"]*b[\\\\\\\\'\\\"]*y(?:[\\\\\\\\'\\\"]*(?:1(?:[\\\\\\\\'\\\"]*[89])?|2[\\\\\\\\'\\\"]*[012]))?|(?:a[\\\\\\\\'\\\"]*r|c[\\\\\\\\'\\\"]*p|p[\\\\\\\\'\\\"]*m)[\\\\\\\\'\\\"]*(?:\\s|<|>).*|n[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*o|o[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*e|s[\\\\\\\\'\\\"]*y[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*c)|t[\\\\\\\\'\\\"]*(?:c[\\\\\\\\'\\\"]*(?:p[\\\\\\\\'\\\"]*(?:t[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*e|i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*g)|s[\\\\\\\\'\\\"]*h)|r[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*e(?:[\\\\\\\\'\\\"]*6)?|e[\\\\\\\\'\\\"]*(?:l[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*t|e[\\\\\\\\'\\\"]*(?:\\s|<|>).*)|i[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*(?:o[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*t|(?:\\s|<|>).*)|a[\\\\\\\\'\\\"]*(?:i[\\\\\\\\'\\\"]*l(?:[\\\\\\\\'\\\"]*f)?|r[\\\\\\\\'\\\"]*(?:\\s|<|>).*)|o[\\\\\\\\'\\\"]*(?:u[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*(?:\\s|<|>).*|p))|u[\\\\\\\\'\\\"]*(?:n[\\\\\\\\'\\\"]*(?:l[\\\\\\\\'\\\"]*(?:i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*k[\\\\\\\\'\\\"]*(?:\\s|<|>).*|z[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*a)|c[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s|a[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*e|r[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*r|s[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*t|z[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*p|x[\\\\\\\\'\\\"]*z)|s[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*(?:(?:a[\\\\\\\\'\\\"]*d|m[\\\\\\\\'\\\"]*o)[\\\\\\\\'\\\"]*d|d[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*l)|l[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*(?:\\s|<|>).*)|m[\\\\\\\\'\\\"]*(?:y[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*q[\\\\\\\\'\\\"]*l(?:[\\\\\\\\'\\\"]*(?:d[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*p(?:[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*w)?|h[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*y|a[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n|s[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*w))?|(?:(?:o[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*n|u[\\\\\\\\'\\\"]*t)[\\\\\\\\'\\\"]*t|v)[\\\\\\\\'\\\"]*(?:\\s|<|>).*)|x[\\\\\\\\'\\\"]*(?:z[\\\\\\\\'\\\"]*(?:(?:[ef][\\\\\\\\'\\\"]*)?g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p|d[\\\\\\\\'\\\"]*(?:i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|e[\\\\\\\\'\\\"]*c)|c[\\\\\\\\'\\\"]*(?:a[\\\\\\\\'\\\"]*t|m[\\\\\\\\'\\\"]*p)|l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s|m[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e|(?:\\s|<|>).*)|a[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*g[\\\\\\\\'\\\"]*s|t[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*m|x[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*(?:\\s|<|>).*)|z[\\\\\\\\'\\\"]*(?:(?:[ef][\\\\\\\\'\\\"]*)?g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p|c[\\\\\\\\'\\\"]*(?:a[\\\\\\\\'\\\"]*t|m[\\\\\\\\'\\\"]*p)|d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|i[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*(?:\\s|<|>).*|l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s|m[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e|r[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*n|s[\\\\\\\\'\\\"]*h)|o[\\\\\\\\'\\\"]*(?:p[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*l|n[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*r)|w[\\\\\\\\'\\\"]*(?:h[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*(?:a[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*i|(?:\\s|<|>).*)|g[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*t|3[\\\\\\\\'\\\"]*m)|v[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*(?:m[\\\\\\\\'\\\"]*(?:\\s|<|>).*|g[\\\\\\\\'\\\"]*r|p[\\\\\\\\'\\\"]*w)|y[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*m)\\b",

/* ********************************** */

/**
 * [ Windows command injection ]
 *
 * This regex detects Windows shell command injections.
 *
 * A command injection takes a form such as:
 *
 *   foo.jpg&ver /r
 *   foo.jpg|ver /r
 *
 * The vulnerability exists when an application executes a shell command
 * without proper input escaping/validation.
 *
 * To prevent false positives, we look for a 'starting sequence' that
 * precedes a command in CMD syntax, such as: ; | & `
 *
 * Anatomy of the regexp:
 *
 * 1. Starting tokens
 *
 * ;        ;cmd
 * \{       {cmd
 * \|       |cmd
 * \|\|     ||cmd
 * &        &cmd
 * &&       &&cmd
 * \n       \ncmd
 * \r       \rcmd
 * `        `cmd
 *
 * 2. Command prefixes
 *
 * (        (cmd)
 * ,        ,cmd
 * @        @cmd
 * '        'cmd'
 * "        "cmd"
 * \s       spacing+cmd
 *
 * 3. Paths
 *
 * [\w'\"\./]+/                          /path/cmd
 * [\\\\'\"\^]*\w[\\\\'\"\^]*:.*\\\\     C:\Program Files\cmd
 * [\^\.\w '\"/\\\\]*\\\\)?[\"\^]*       \\net\share\dir\cmd
 *
 * 4. Quoting
 *
 * \"       "cmd"
 * \^       ^cmd
 *
 * 5. Extension/switches
 *
 * \.[\"\^]*\w+      cmd.com, cmd.exe, etc.
 * /b                cmd/h
 *
 * An effort is made to combat evasions by CMD syntax; for example,
 * the following strings are valid: c^md, @cmd, "c"md.
 *
 * This regex is case-insensitive.
 */

  "(?i)(?:;|\\{|\\||\\|\\||&|&&|\\n|\\r|`)\\s*[\\(,@\\'\\\"\\s]*(?:[\\w'\\\"\\./]+/|[\\\\\\\\'\\\"\\^]*\\w[\\\\\\\\'\\\"\\^]*:.*\\\\\\\\|[\\^\\.\\w '\\\"/\\\\\\\\]*\\\\\\\\)?[\\\"\\^]*(?:m[\\\"\\^]*(?:y[\\\"\\^]*s[\\\"\\^]*q[\\\"\\^]*l(?:[\\\"\\^]*(?:d[\\\"\\^]*u[\\\"\\^]*m[\\\"\\^]*p(?:[\\\"\\^]*s[\\\"\\^]*l[\\\"\\^]*o[\\\"\\^]*w)?|h[\\\"\\^]*o[\\\"\\^]*t[\\\"\\^]*c[\\\"\\^]*o[\\\"\\^]*p[\\\"\\^]*y|a[\\\"\\^]*d[\\\"\\^]*m[\\\"\\^]*i[\\\"\\^]*n|s[\\\"\\^]*h[\\\"\\^]*o[\\\"\\^]*w))?|s[\\\"\\^]*(?:i[\\\"\\^]*(?:n[\\\"\\^]*f[\\\"\\^]*o[\\\"\\^]*3[\\\"\\^]*2|e[\\\"\\^]*x[\\\"\\^]*e[\\\"\\^]*c)|c[\\\"\\^]*o[\\\"\\^]*n[\\\"\\^]*f[\\\"\\^]*i[\\\"\\^]*g|g[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|t[\\\"\\^]*s[\\\"\\^]*c)|o[\\\"\\^]*(?:u[\\\"\\^]*n[\\\"\\^]*t[\\\"\\^]*(?:(?:[\\s,;]|\\.|/|<|>).*|v[\\\"\\^]*o[\\\"\\^]*l)|v[\\\"\\^]*e[\\\"\\^]*u[\\\"\\^]*s[\\\"\\^]*e[\\\"\\^]*r|[dr][\\\"\\^]*e[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*)|k[\\\"\\^]*(?:d[\\\"\\^]*i[\\\"\\^]*r[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|l[\\\"\\^]*i[\\\"\\^]*n[\\\"\\^]*k)|d[\\\"\\^]*(?:s[\\\"\\^]*c[\\\"\\^]*h[\\\"\\^]*e[\\\"\\^]*d|(?:[\\s,;]|\\.|/|<|>).*)|a[\\\"\\^]*p[\\\"\\^]*i[\\\"\\^]*s[\\\"\\^]*e[\\\"\\^]*n[\\\"\\^]*d|b[\\\"\\^]*s[\\\"\\^]*a[\\\"\\^]*c[\\\"\\^]*l[\\\"\\^]*i|e[\\\"\\^]*a[\\\"\\^]*s[\\\"\\^]*u[\\\"\\^]*r[\\\"\\^]*e|m[\\\"\\^]*s[\\\"\\^]*y[\\\"\\^]*s)|d[\\\"\\^]*(?:i[\\\"\\^]*(?:s[\\\"\\^]*k[\\\"\\^]*(?:(?:m[\\\"\\^]*g[\\\"\\^]*m|p[\\\"\\^]*a[\\\"\\^]*r)[\\\"\\^]*t|s[\\\"\\^]*h[\\\"\\^]*a[\\\"\\^]*d[\\\"\\^]*o[\\\"\\^]*w)|r[\\\"\\^]*(?:(?:[\\s,;]|\\.|/|<|>).*|u[\\\"\\^]*s[\\\"\\^]*e)|f[\\\"\\^]*f[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*)|e[\\\"\\^]*(?:l[\\\"\\^]*(?:p[\\\"\\^]*r[\\\"\\^]*o[\\\"\\^]*f|t[\\\"\\^]*r[\\\"\\^]*e[\\\"\\^]*e|(?:[\\s,;]|\\.|/|<|>).*)|v[\\\"\\^]*(?:m[\\\"\\^]*g[\\\"\\^]*m[\\\"\\^]*t|c[\\\"\\^]*o[\\\"\\^]*n)|(?:f[\\\"\\^]*r[\\\"\\^]*a|b[\\\"\\^]*u)[\\\"\\^]*g)|s[\\\"\\^]*(?:a[\\\"\\^]*(?:c[\\\"\\^]*l[\\\"\\^]*s|d[\\\"\\^]*d)|q[\\\"\\^]*u[\\\"\\^]*e[\\\"\\^]*r[\\\"\\^]*y|m[\\\"\\^]*o[\\\"\\^]*(?:v[\\\"\\^]*e|d)|g[\\\"\\^]*e[\\\"\\^]*t|r[\\\"\\^]*m)|(?:r[\\\"\\^]*i[\\\"\\^]*v[\\\"\\^]*e[\\\"\\^]*r[\\\"\\^]*q[\\\"\\^]*u[\\\"\\^]*e[\\\"\\^]*r|o[\\\"\\^]*s[\\\"\\^]*k[\\\"\\^]*e)[\\\"\\^]*y|(?:c[\\\"\\^]*o[\\\"\\^]*m[\\\"\\^]*c[\\\"\\^]*n[\\\"\\^]*f|x[\\\"\\^]*d[\\\"\\^]*i[\\\"\\^]*a)[\\\"\\^]*g|a[\\\"\\^]*t[\\\"\\^]*e[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|n[\\\"\\^]*s[\\\"\\^]*s[\\\"\\^]*t[\\\"\\^]*a[\\\"\\^]*t)|c[\\\"\\^]*(?:o[\\\"\\^]*(?:m[\\\"\\^]*(?:p[\\\"\\^]*(?:(?:a[\\\"\\^]*c[\\\"\\^]*t[\\\"\\^]*)?(?:[\\s,;]|\\.|/|<|>).*|m[\\\"\\^]*g[\\\"\\^]*m[\\\"\\^]*t)|e[\\\"\\^]*x[\\\"\\^]*p)|n[\\\"\\^]*(?:2[\\\"\\^]*p|v[\\\"\\^]*e)[\\\"\\^]*r[\\\"\\^]*t|p[\\\"\\^]*y)|l[\\\"\\^]*(?:e[\\\"\\^]*a[\\\"\\^]*(?:n[\\\"\\^]*m[\\\"\\^]*g[\\\"\\^]*r|r[\\\"\\^]*m[\\\"\\^]*e[\\\"\\^]*m)|u[\\\"\\^]*s[\\\"\\^]*t[\\\"\\^]*e[\\\"\\^]*r)|h[\\\"\\^]*(?:k[\\\"\\^]*(?:n[\\\"\\^]*t[\\\"\\^]*f[\\\"\\^]*s|d[\\\"\\^]*s[\\\"\\^]*k)|d[\\\"\\^]*i[\\\"\\^]*r[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*)|s[\\\"\\^]*(?:c[\\\"\\^]*(?:r[\\\"\\^]*i[\\\"\\^]*p[\\\"\\^]*t|c[\\\"\\^]*m[\\\"\\^]*d)|v[\\\"\\^]*d[\\\"\\^]*e)|e[\\\"\\^]*r[\\\"\\^]*t[\\\"\\^]*(?:u[\\\"\\^]*t[\\\"\\^]*i[\\\"\\^]*l|r[\\\"\\^]*e[\\\"\\^]*q)|a[\\\"\\^]*(?:l[\\\"\\^]*l[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|c[\\\"\\^]*l[\\\"\\^]*s)|m[\\\"\\^]*d(?:[\\\"\\^]*k[\\\"\\^]*e[\\\"\\^]*y)?|i[\\\"\\^]*p[\\\"\\^]*h[\\\"\\^]*e[\\\"\\^]*r|u[\\\"\\^]*r[\\\"\\^]*l)|f[\\\"\\^]*(?:o[\\\"\\^]*r[\\\"\\^]*(?:m[\\\"\\^]*a[\\\"\\^]*t[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|f[\\\"\\^]*i[\\\"\\^]*l[\\\"\\^]*e[\\\"\\^]*s|e[\\\"\\^]*a[\\\"\\^]*c[\\\"\\^]*h)|i[\\\"\\^]*n[\\\"\\^]*d[\\\"\\^]*(?:(?:[\\s,;]|\\.|/|<|>).*|s[\\\"\\^]*t[\\\"\\^]*r)|s[\\\"\\^]*(?:m[\\\"\\^]*g[\\\"\\^]*m[\\\"\\^]*t|u[\\\"\\^]*t[\\\"\\^]*i[\\\"\\^]*l)|t[\\\"\\^]*(?:p[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|y[\\\"\\^]*p[\\\"\\^]*e)|r[\\\"\\^]*e[\\\"\\^]*e[\\\"\\^]*d[\\\"\\^]*i[\\\"\\^]*s[\\\"\\^]*k|c[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|g[\\\"\\^]*r[\\\"\\^]*e[\\\"\\^]*p)|n[\\\"\\^]*(?:e[\\\"\\^]*t[\\\"\\^]*(?:s[\\\"\\^]*(?:t[\\\"\\^]*a[\\\"\\^]*t|v[\\\"\\^]*c|h)|(?:[\\s,;]|\\.|/|<|>).*|c[\\\"\\^]*a[\\\"\\^]*t|d[\\\"\\^]*o[\\\"\\^]*m)|t[\\\"\\^]*(?:b[\\\"\\^]*a[\\\"\\^]*c[\\\"\\^]*k[\\\"\\^]*u[\\\"\\^]*p|r[\\\"\\^]*i[\\\"\\^]*g[\\\"\\^]*h[\\\"\\^]*t[\\\"\\^]*s)|(?:s[\\\"\\^]*l[\\\"\\^]*o[\\\"\\^]*o[\\\"\\^]*k[\\\"\\^]*u|m[\\\"\\^]*a)[\\\"\\^]*p|c[\\\"\\^]*(?:(?:[\\s,;]|\\.|/|<|>).*|a[\\\"\\^]*t)|b[\\\"\\^]*t[\\\"\\^]*s[\\\"\\^]*t[\\\"\\^]*a[\\\"\\^]*t)|e[\\\"\\^]*(?:x[\\\"\\^]*p[\\\"\\^]*(?:a[\\\"\\^]*n[\\\"\\^]*d[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|l[\\\"\\^]*o[\\\"\\^]*r[\\\"\\^]*e[\\\"\\^]*r)|v[\\\"\\^]*e[\\\"\\^]*n[\\\"\\^]*t[\\\"\\^]*(?:c[\\\"\\^]*r[\\\"\\^]*e[\\\"\\^]*a[\\\"\\^]*t[\\\"\\^]*e|v[\\\"\\^]*w[\\\"\\^]*r)|n[\\\"\\^]*d[\\\"\\^]*l[\\\"\\^]*o[\\\"\\^]*c[\\\"\\^]*a[\\\"\\^]*l|g[\\\"\\^]*r[\\\"\\^]*e[\\\"\\^]*p|r[\\\"\\^]*a[\\\"\\^]*s[\\\"\\^]*e|c[\\\"\\^]*h[\\\"\\^]*o)|g[\\\"\\^]*(?:a[\\\"\\^]*t[\\\"\\^]*h[\\\"\\^]*e[\\\"\\^]*r[\\\"\\^]*n[\\\"\\^]*e[\\\"\\^]*t[\\\"\\^]*w[\\\"\\^]*o[\\\"\\^]*r[\\\"\\^]*k[\\\"\\^]*i[\\\"\\^]*n[\\\"\\^]*f[\\\"\\^]*o|p[\\\"\\^]*(?:(?:r[\\\"\\^]*e[\\\"\\^]*s[\\\"\\^]*u[\\\"\\^]*l|e[\\\"\\^]*d[\\\"\\^]*i)[\\\"\\^]*t|u[\\\"\\^]*p[\\\"\\^]*d[\\\"\\^]*a[\\\"\\^]*t[\\\"\\^]*e)|i[\\\"\\^]*t[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|e[\\\"\\^]*t[\\\"\\^]*m[\\\"\\^]*a[\\\"\\^]*c)|i[\\\"\\^]*(?:r[\\\"\\^]*b(?:[\\\"\\^]*(?:1(?:[\\\"\\^]*[89])?|2[\\\"\\^]*[012]))?|f[\\\"\\^]*m[\\\"\\^]*e[\\\"\\^]*m[\\\"\\^]*b[\\\"\\^]*e[\\\"\\^]*r|p[\\\"\\^]*c[\\\"\\^]*o[\\\"\\^]*n[\\\"\\^]*f[\\\"\\^]*i[\\\"\\^]*g|n[\\\"\\^]*e[\\\"\\^]*t[\\\"\\^]*c[\\\"\\^]*p[\\\"\\^]*l|c[\\\"\\^]*a[\\\"\\^]*c[\\\"\\^]*l[\\\"\\^]*s)|a[\\\"\\^]*(?:d[\\\"\\^]*(?:d[\\\"\\^]*u[\\\"\\^]*s[\\\"\\^]*e[\\\"\\^]*r[\\\"\\^]*s|m[\\\"\\^]*o[\\\"\\^]*d[\\\"\\^]*c[\\\"\\^]*m[\\\"\\^]*d)|r[\\\"\\^]*p[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|t[\\\"\\^]*t[\\\"\\^]*r[\\\"\\^]*i[\\\"\\^]*b|s[\\\"\\^]*s[\\\"\\^]*o[\\\"\\^]*c|z[\\\"\\^]*m[\\\"\\^]*a[\\\"\\^]*n)|l[\\\"\\^]*(?:o[\\\"\\^]*g[\\\"\\^]*(?:e[\\\"\\^]*v[\\\"\\^]*e[\\\"\\^]*n[\\\"\\^]*t|t[\\\"\\^]*i[\\\"\\^]*m[\\\"\\^]*e|m[\\\"\\^]*a[\\\"\\^]*n|o[\\\"\\^]*f[\\\"\\^]*f)|a[\\\"\\^]*b[\\\"\\^]*e[\\\"\\^]*l[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|u[\\\"\\^]*s[\\\"\\^]*r[\\\"\\^]*m[\\\"\\^]*g[\\\"\\^]*r)|b[\\\"\\^]*(?:(?:c[\\\"\\^]*d[\\\"\\^]*(?:b[\\\"\\^]*o[\\\"\\^]*o|e[\\\"\\^]*d[\\\"\\^]*i)|r[\\\"\\^]*o[\\\"\\^]*w[\\\"\\^]*s[\\\"\\^]*t[\\\"\\^]*a)[\\\"\\^]*t|i[\\\"\\^]*t[\\\"\\^]*s[\\\"\\^]*a[\\\"\\^]*d[\\\"\\^]*m[\\\"\\^]*i[\\\"\\^]*n|o[\\\"\\^]*o[\\\"\\^]*t[\\\"\\^]*c[\\\"\\^]*f[\\\"\\^]*g)|h[\\\"\\^]*(?:o[\\\"\\^]*s[\\\"\\^]*t[\\\"\\^]*n[\\\"\\^]*a[\\\"\\^]*m[\\\"\\^]*e|d[\\\"\\^]*w[\\\"\\^]*w[\\\"\\^]*i[\\\"\\^]*z)|j[\\\"\\^]*a[\\\"\\^]*v[\\\"\\^]*a[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|7[\\\"\\^]*z(?:[\\\"\\^]*[ar])?)(?:\\.[\\\"\\^]*\\w+)?\\b",

/* ********************************** */

/**
 *This regex is also triggered by an Oracle WebLogic Remote Command Execution exploit:
 *[ Oracle WebLogic vulnerability CVE-2017-10271 - Exploit tested: https://www.exploit-db.com/exploits/43458 ]
 */

  "(?i)(?:;|\\{|\\||\\|\\||&|&&|\\n|\\r|`)\\s*[\\(,@\\'\\\"\\s]*(?:[\\w'\\\"\\./]+/|[\\\\\\\\'\\\"\\^]*\\w[\\\\\\\\'\\\"\\^]*:.*\\\\\\\\|[\\^\\.\\w '\\\"/\\\\\\\\]*\\\\\\\\)?[\\\"\\^]*(?:s[\\\"\\^]*(?:y[\\\"\\^]*s[\\\"\\^]*(?:t[\\\"\\^]*e[\\\"\\^]*m[\\\"\\^]*(?:p[\\\"\\^]*r[\\\"\\^]*o[\\\"\\^]*p[\\\"\\^]*e[\\\"\\^]*r[\\\"\\^]*t[\\\"\\^]*i[\\\"\\^]*e[\\\"\\^]*s[\\\"\\^]*(?:d[\\\"\\^]*a[\\\"\\^]*t[\\\"\\^]*a[\\\"\\^]*e[\\\"\\^]*x[\\\"\\^]*e[\\\"\\^]*c[\\\"\\^]*u[\\\"\\^]*t[\\\"\\^]*i[\\\"\\^]*o[\\\"\\^]*n[\\\"\\^]*p[\\\"\\^]*r[\\\"\\^]*e[\\\"\\^]*v[\\\"\\^]*e[\\\"\\^]*n[\\\"\\^]*t[\\\"\\^]*i[\\\"\\^]*o[\\\"\\^]*n|(?:p[\\\"\\^]*e[\\\"\\^]*r[\\\"\\^]*f[\\\"\\^]*o[\\\"\\^]*r[\\\"\\^]*m[\\\"\\^]*a[\\\"\\^]*n[\\\"\\^]*c|h[\\\"\\^]*a[\\\"\\^]*r[\\\"\\^]*d[\\\"\\^]*w[\\\"\\^]*a[\\\"\\^]*r)[\\\"\\^]*e|a[\\\"\\^]*d[\\\"\\^]*v[\\\"\\^]*a[\\\"\\^]*n[\\\"\\^]*c[\\\"\\^]*e[\\\"\\^]*d)|i[\\\"\\^]*n[\\\"\\^]*f[\\\"\\^]*o)|k[\\\"\\^]*e[\\\"\\^]*y|d[\\\"\\^]*m)|h[\\\"\\^]*(?:o[\\\"\\^]*(?:w[\\\"\\^]*(?:g[\\\"\\^]*r[\\\"\\^]*p|m[\\\"\\^]*b[\\\"\\^]*r)[\\\"\\^]*s|r[\\\"\\^]*t[\\\"\\^]*c[\\\"\\^]*u[\\\"\\^]*t)|e[\\\"\\^]*l[\\\"\\^]*l[\\\"\\^]*r[\\\"\\^]*u[\\\"\\^]*n[\\\"\\^]*a[\\\"\\^]*s|u[\\\"\\^]*t[\\\"\\^]*d[\\\"\\^]*o[\\\"\\^]*w[\\\"\\^]*n|r[\\\"\\^]*p[\\\"\\^]*u[\\\"\\^]*b[\\\"\\^]*w|a[\\\"\\^]*r[\\\"\\^]*e|i[\\\"\\^]*f[\\\"\\^]*t)|e[\\\"\\^]*(?:t[\\\"\\^]*(?:(?:x[\\\"\\^]*)?(?:[\\s,;]|\\.|/|<|>).*|l[\\\"\\^]*o[\\\"\\^]*c[\\\"\\^]*a[\\\"\\^]*l)|c[\\\"\\^]*p[\\\"\\^]*o[\\\"\\^]*l|l[\\\"\\^]*e[\\\"\\^]*c[\\\"\\^]*t)|c[\\\"\\^]*(?:h[\\\"\\^]*t[\\\"\\^]*a[\\\"\\^]*s[\\\"\\^]*k[\\\"\\^]*s|l[\\\"\\^]*i[\\\"\\^]*s[\\\"\\^]*t)|u[\\\"\\^]*b[\\\"\\^]*(?:i[\\\"\\^]*n[\\\"\\^]*a[\\\"\\^]*c[\\\"\\^]*l|s[\\\"\\^]*t)|t[\\\"\\^]*a[\\\"\\^]*r[\\\"\\^]*t[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|i[\\\"\\^]*g[\\\"\\^]*v[\\\"\\^]*e[\\\"\\^]*r[\\\"\\^]*i[\\\"\\^]*f|l[\\\"\\^]*(?:e[\\\"\\^]*e[\\\"\\^]*p|m[\\\"\\^]*g[\\\"\\^]*r)|o[\\\"\\^]*r[\\\"\\^]*t|f[\\\"\\^]*c|v[\\\"\\^]*n)|p[\\\"\\^]*(?:s[\\\"\\^]*(?:s[\\\"\\^]*(?:h[\\\"\\^]*u[\\\"\\^]*t[\\\"\\^]*d[\\\"\\^]*o[\\\"\\^]*w[\\\"\\^]*n|e[\\\"\\^]*r[\\\"\\^]*v[\\\"\\^]*i[\\\"\\^]*c[\\\"\\^]*e|u[\\\"\\^]*s[\\\"\\^]*p[\\\"\\^]*e[\\\"\\^]*n[\\\"\\^]*d)|l[\\\"\\^]*(?:o[\\\"\\^]*g[\\\"\\^]*(?:g[\\\"\\^]*e[\\\"\\^]*d[\\\"\\^]*o[\\\"\\^]*n|l[\\\"\\^]*i[\\\"\\^]*s[\\\"\\^]*t)|i[\\\"\\^]*s[\\\"\\^]*t)|p[\\\"\\^]*(?:a[\\\"\\^]*s[\\\"\\^]*s[\\\"\\^]*w[\\\"\\^]*d|i[\\\"\\^]*n[\\\"\\^]*g)|g[\\\"\\^]*e[\\\"\\^]*t[\\\"\\^]*s[\\\"\\^]*i[\\\"\\^]*d|e[\\\"\\^]*x[\\\"\\^]*e[\\\"\\^]*c|f[\\\"\\^]*i[\\\"\\^]*l[\\\"\\^]*e|i[\\\"\\^]*n[\\\"\\^]*f[\\\"\\^]*o|k[\\\"\\^]*i[\\\"\\^]*l[\\\"\\^]*l)|o[\\\"\\^]*(?:w[\\\"\\^]*e[\\\"\\^]*r[\\\"\\^]*(?:s[\\\"\\^]*h[\\\"\\^]*e[\\\"\\^]*l[\\\"\\^]*l(?:[\\\"\\^]*_[\\\"\\^]*i[\\\"\\^]*s[\\\"\\^]*e)?|c[\\\"\\^]*f[\\\"\\^]*g)|r[\\\"\\^]*t[\\\"\\^]*q[\\\"\\^]*r[\\\"\\^]*y|p[\\\"\\^]*d)|r[\\\"\\^]*(?:i[\\\"\\^]*n[\\\"\\^]*t[\\\"\\^]*(?:(?:[\\s,;]|\\.|/|<|>).*|b[\\\"\\^]*r[\\\"\\^]*m)|n[\\\"\\^]*(?:c[\\\"\\^]*n[\\\"\\^]*f[\\\"\\^]*g|m[\\\"\\^]*n[\\\"\\^]*g[\\\"\\^]*r)|o[\\\"\\^]*m[\\\"\\^]*p[\\\"\\^]*t)|a[\\\"\\^]*t[\\\"\\^]*h[\\\"\\^]*(?:p[\\\"\\^]*i[\\\"\\^]*n[\\\"\\^]*g|(?:[\\s,;]|\\.|/|<|>).*)|e[\\\"\\^]*r[\\\"\\^]*(?:l(?:[\\\"\\^]*(?:s[\\\"\\^]*h|5))?|f[\\\"\\^]*m[\\\"\\^]*o[\\\"\\^]*n)|y[\\\"\\^]*t[\\\"\\^]*h[\\\"\\^]*o[\\\"\\^]*n(?:[\\\"\\^]*(?:3(?:[\\\"\\^]*m)?|2))?|k[\\\"\\^]*g[\\\"\\^]*m[\\\"\\^]*g[\\\"\\^]*r|h[\\\"\\^]*p(?:[\\\"\\^]*[57])?|u[\\\"\\^]*s[\\\"\\^]*h[\\\"\\^]*d|i[\\\"\\^]*n[\\\"\\^]*g)|r[\\\"\\^]*(?:e[\\\"\\^]*(?:(?:p[\\\"\\^]*l[\\\"\\^]*a[\\\"\\^]*c[\\\"\\^]*e|n(?:[\\\"\\^]*a[\\\"\\^]*m[\\\"\\^]*e)?|s[\\\"\\^]*e[\\\"\\^]*t)[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|g[\\\"\\^]*(?:s[\\\"\\^]*v[\\\"\\^]*r[\\\"\\^]*3[\\\"\\^]*2|e[\\\"\\^]*d[\\\"\\^]*i[\\\"\\^]*t|(?:[\\s,;]|\\.|/|<|>).*|i[\\\"\\^]*n[\\\"\\^]*i)|c[\\\"\\^]*(?:d[\\\"\\^]*i[\\\"\\^]*s[\\\"\\^]*c|o[\\\"\\^]*v[\\\"\\^]*e[\\\"\\^]*r)|k[\\\"\\^]*e[\\\"\\^]*y[\\\"\\^]*w[\\\"\\^]*i[\\\"\\^]*z)|u[\\\"\\^]*(?:n[\\\"\\^]*(?:d[\\\"\\^]*l[\\\"\\^]*l[\\\"\\^]*3[\\\"\\^]*2|a[\\\"\\^]*s)|b[\\\"\\^]*y[\\\"\\^]*(?:1(?:[\\\"\\^]*[89])?|2[\\\"\\^]*[012]))|a[\\\"\\^]*(?:s[\\\"\\^]*(?:p[\\\"\\^]*h[\\\"\\^]*o[\\\"\\^]*n[\\\"\\^]*e|d[\\\"\\^]*i[\\\"\\^]*a[\\\"\\^]*l)|r[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*)|m[\\\"\\^]*(?:(?:d[\\\"\\^]*i[\\\"\\^]*r[\\\"\\^]*)?(?:[\\s,;]|\\.|/|<|>).*|t[\\\"\\^]*s[\\\"\\^]*h[\\\"\\^]*a[\\\"\\^]*r[\\\"\\^]*e)|o[\\\"\\^]*(?:u[\\\"\\^]*t[\\\"\\^]*e[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|b[\\\"\\^]*o[\\\"\\^]*c[\\\"\\^]*o[\\\"\\^]*p[\\\"\\^]*y)|s[\\\"\\^]*(?:t[\\\"\\^]*r[\\\"\\^]*u[\\\"\\^]*i|y[\\\"\\^]*n[\\\"\\^]*c)|d[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*)|t[\\\"\\^]*(?:a[\\\"\\^]*(?:s[\\\"\\^]*k[\\\"\\^]*(?:k[\\\"\\^]*i[\\\"\\^]*l[\\\"\\^]*l|l[\\\"\\^]*i[\\\"\\^]*s[\\\"\\^]*t|s[\\\"\\^]*c[\\\"\\^]*h[\\\"\\^]*d|m[\\\"\\^]*g[\\\"\\^]*r)|k[\\\"\\^]*e[\\\"\\^]*o[\\\"\\^]*w[\\\"\\^]*n)|(?:i[\\\"\\^]*m[\\\"\\^]*e[\\\"\\^]*o[\\\"\\^]*u|p[\\\"\\^]*m[\\\"\\^]*i[\\\"\\^]*n[\\\"\\^]*i|e[\\\"\\^]*l[\\\"\\^]*n[\\\"\\^]*e|l[\\\"\\^]*i[\\\"\\^]*s)[\\\"\\^]*t|s[\\\"\\^]*(?:d[\\\"\\^]*i[\\\"\\^]*s[\\\"\\^]*c[\\\"\\^]*o|s[\\\"\\^]*h[\\\"\\^]*u[\\\"\\^]*t[\\\"\\^]*d)[\\\"\\^]*n|y[\\\"\\^]*p[\\\"\\^]*e[\\\"\\^]*(?:p[\\\"\\^]*e[\\\"\\^]*r[\\\"\\^]*f|(?:[\\s,;]|\\.|/|<|>).*)|r[\\\"\\^]*(?:a[\\\"\\^]*c[\\\"\\^]*e[\\\"\\^]*r[\\\"\\^]*t|e[\\\"\\^]*e))|w[\\\"\\^]*(?:i[\\\"\\^]*n[\\\"\\^]*(?:d[\\\"\\^]*i[\\\"\\^]*f[\\\"\\^]*f|m[\\\"\\^]*s[\\\"\\^]*d[\\\"\\^]*p|v[\\\"\\^]*a[\\\"\\^]*r|r[\\\"\\^]*[ms])|u[\\\"\\^]*(?:a[\\\"\\^]*(?:u[\\\"\\^]*c[\\\"\\^]*l[\\\"\\^]*t|p[\\\"\\^]*p)|s[\\\"\\^]*a)|s[\\\"\\^]*c[\\\"\\^]*(?:r[\\\"\\^]*i[\\\"\\^]*p[\\\"\\^]*t|u[\\\"\\^]*i)|e[\\\"\\^]*v[\\\"\\^]*t[\\\"\\^]*u[\\\"\\^]*t[\\\"\\^]*i[\\\"\\^]*l|m[\\\"\\^]*i[\\\"\\^]*(?:m[\\\"\\^]*g[\\\"\\^]*m[\\\"\\^]*t|c)|a[\\\"\\^]*i[\\\"\\^]*t[\\\"\\^]*f[\\\"\\^]*o[\\\"\\^]*r|h[\\\"\\^]*o[\\\"\\^]*a[\\\"\\^]*m[\\\"\\^]*i|g[\\\"\\^]*e[\\\"\\^]*t)|u[\\\"\\^]*(?:s[\\\"\\^]*(?:e[\\\"\\^]*r[\\\"\\^]*a[\\\"\\^]*c[\\\"\\^]*c[\\\"\\^]*o[\\\"\\^]*u[\\\"\\^]*n[\\\"\\^]*t[\\\"\\^]*c[\\\"\\^]*o[\\\"\\^]*n[\\\"\\^]*t[\\\"\\^]*r[\\\"\\^]*o[\\\"\\^]*l[\\\"\\^]*s[\\\"\\^]*e[\\\"\\^]*t[\\\"\\^]*t[\\\"\\^]*i[\\\"\\^]*n[\\\"\\^]*g[\\\"\\^]*s|r[\\\"\\^]*s[\\\"\\^]*t[\\\"\\^]*a[\\\"\\^]*t)|n[\\\"\\^]*(?:r[\\\"\\^]*a[\\\"\\^]*r|z[\\\"\\^]*i[\\\"\\^]*p))|q[\\\"\\^]*(?:u[\\\"\\^]*e[\\\"\\^]*r[\\\"\\^]*y[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|p[\\\"\\^]*r[\\\"\\^]*o[\\\"\\^]*c[\\\"\\^]*e[\\\"\\^]*s[\\\"\\^]*s|w[\\\"\\^]*i[\\\"\\^]*n[\\\"\\^]*s[\\\"\\^]*t[\\\"\\^]*a|g[\\\"\\^]*r[\\\"\\^]*e[\\\"\\^]*p)|o[\\\"\\^]*(?:d[\\\"\\^]*b[\\\"\\^]*c[\\\"\\^]*(?:a[\\\"\\^]*d[\\\"\\^]*3[\\\"\\^]*2|c[\\\"\\^]*o[\\\"\\^]*n[\\\"\\^]*f)|p[\\\"\\^]*e[\\\"\\^]*n[\\\"\\^]*f[\\\"\\^]*i[\\\"\\^]*l[\\\"\\^]*e[\\\"\\^]*s)|v[\\\"\\^]*(?:o[\\\"\\^]*l[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*|e[\\\"\\^]*r[\\\"\\^]*i[\\\"\\^]*f[\\\"\\^]*y)|x[\\\"\\^]*c[\\\"\\^]*(?:a[\\\"\\^]*c[\\\"\\^]*l[\\\"\\^]*s|o[\\\"\\^]*p[\\\"\\^]*y)|z[\\\"\\^]*i[\\\"\\^]*p[\\\"\\^]*(?:[\\s,;]|\\.|/|<|>).*)(?:\\.[\\\"\\^]*\\w+)?\\b",

/* ********************************** */

/**
 * [ Unix shell expressions ]
 *
 * Detects the following patterns which are common in Unix shell scripts
 * and oneliners:
 *
 * $(foo)    Command substitution
 * ${foo}    Parameter expansion
 * <(foo)    Process substitution
 * >(foo)    Process substitution
 * $((foo))  Arithmetic expansion
 */

  "(?:\\$(?:\\((?:\\(.*\\)|.*)\\)|\\{.*\\})|[<>]\\(.*\\))",

/* ********************************** */

/**
 * [ Windows FOR, IF commands ]
 *
 * This regex detects Windows command shell FOR and IF commands.
 *
 * Examples:
 *
 *   FOR              %a IN (set) DO
 *   FOR /D           %a IN (dirs) DO
 *   FOR /F "options" %a IN (text|"text") DO
 *   FOR /L           %a IN (start,step,end) DO
 *   FOR /R C:\dir    %A IN (set) DO
 *
 *   IF [/I] [NOT] EXIST filename | DEFINED define | ERRORLEVEL n | CMDEXTVERSION n
 *   IF [/I] [NOT] item1   [==|EQU|NEQ|LSS|LEQ|GTR|GEQ] item2
 *   IF [/I] [NOT] (item1) [==|EQU|NEQ|LSS|LEQ|GTR|GEQ] (item2)
 *
 * http://ss64.com/nt/if.html
 * http://ss64.com/nt/for.html
 */

  "\\b(?:if(?:/i)?(?: not)?(?: exist\\b| defined\\b| errorlevel\\b| cmdextversion\\b|(?: |\\().*(?:\\bgeq\\b|\\bequ\\b|\\bneq\\b|\\bleq\\b|\\bgtr\\b|\\blss\\b|==))|for(?:/[dflr].*)? %+[^ ]+ in\\(.*\\)\\s?do)",

/* ********************************** */

/**
 * [ Unix direct remote command execution ]
 *
 * Detects Unix commands at the start of a parameter (direct RCE).
 * Example: foo=wget%20www.example.com
 *
 * This case is different from command injection, where a
 * command string is appended (injected) to a regular parameter, and then
 * passed to a shell unescaped.
 *
 * This regex is also triggered by an Oracle WebLogic Remote Command Execution exploit:
 * [ Oracle WebLogic vulnerability CVE-2017-10271 - Exploit tested: https://www.exploit-db.com/exploits/43458 ]
 */

  "(?:^|=)\\s*(?:{|\\s*\\(\\s*|\\w+=(?:[^\\s]*|\\$.*|\\$.*|<.*|>.*|\\'.*\\'|\\\".*\\\")\\s+|!\\s*|\\$)*\\s*(?:'|\\\")*(?:[\\?\\*\\[\\]\\(\\)\\-\\|+\\w'\\\"\\./\\\\\\\\]+/)?[\\\\\\\\'\\\"]*(?:l[\\\\\\\\'\\\"]*(?:s(?:[\\\\\\\\'\\\"]*(?:b[\\\\\\\\'\\\"]*_[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*e|c[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*u|m[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*d|p[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*i|u[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*b|-[\\\\\\\\'\\\"]*F|o[\\\\\\\\'\\\"]*f))?|z[\\\\\\\\'\\\"]*(?:(?:[ef][\\\\\\\\'\\\"]*)?g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p|c[\\\\\\\\'\\\"]*(?:a[\\\\\\\\'\\\"]*t|m[\\\\\\\\'\\\"]*p)|m[\\\\\\\\'\\\"]*(?:o[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e|a)|d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s)|e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*(?:(?:f[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*l|p[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*p)[\\\\\\\\'\\\"]*e|e[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*o)|a[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*(?:l[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*g(?:[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n)?|c[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*m)|w[\\\\\\\\'\\\"]*p(?:[\\\\\\\\'\\\"]*-[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*w[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*d)?|f[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*p(?:[\\\\\\\\'\\\"]*g[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*t)?|y[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*x)|s[\\\\\\\\'\\\"]*(?:e[\\\\\\\\'\\\"]*(?:t[\\\\\\\\'\\\"]*(?:e[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*v|s[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*d)|n[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*l|d)|h(?:[\\\\\\\\'\\\"]*\\.[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*b)?|o[\\\\\\\\'\\\"]*(?:u[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*e|c[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t)|t[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*g[\\\\\\\\'\\\"]*s|y[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*l|c[\\\\\\\\'\\\"]*(?:h[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*d|p)|d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|f[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*p|u[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*o|s[\\\\\\\\'\\\"]*h|v[\\\\\\\\'\\\"]*n)|p[\\\\\\\\'\\\"]*(?:t[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*r(?:[\\\\\\\\'\\\"]*(?:d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p))?|y[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*n(?:[\\\\\\\\'\\\"]*(?:3(?:[\\\\\\\\'\\\"]*m)?|2))?|k[\\\\\\\\'\\\"]*(?:e[\\\\\\\\'\\\"]*x[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*c|i[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*l)|r[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*v|(?:g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e|f[\\\\\\\\'\\\"]*t)[\\\\\\\\'\\\"]*p|e[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*l(?:[\\\\\\\\'\\\"]*5)?|h[\\\\\\\\'\\\"]*p(?:[\\\\\\\\'\\\"]*[57])?|i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*g|o[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*d)|n[\\\\\\\\'\\\"]*(?:c(?:[\\\\\\\\'\\\"]*(?:\\.[\\\\\\\\'\\\"]*(?:t[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*l|o[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*b[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*d)|a[\\\\\\\\'\\\"]*t))?|e[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*(?:k[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*-[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*p|(?:s[\\\\\\\\'\\\"]*t|c)[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t)|o[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*p|p[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*g|s[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t)|t[\\\\\\\\'\\\"]*(?:c[\\\\\\\\'\\\"]*(?:p[\\\\\\\\'\\\"]*(?:t[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*e|i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*g)|s[\\\\\\\\'\\\"]*h)|r[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*e(?:[\\\\\\\\'\\\"]*6)?|i[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*e(?:[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*t)?|a[\\\\\\\\'\\\"]*(?:i[\\\\\\\\'\\\"]*l(?:[\\\\\\\\'\\\"]*f)?|r)|e[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*t)|r[\\\\\\\\'\\\"]*(?:e[\\\\\\\\'\\\"]*(?:p[\\\\\\\\'\\\"]*(?:l[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*e|e[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t)|a[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*h|n[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*e)|u[\\\\\\\\'\\\"]*b[\\\\\\\\'\\\"]*y(?:[\\\\\\\\'\\\"]*(?:1(?:[\\\\\\\\'\\\"]*[89])?|2[\\\\\\\\'\\\"]*[012]))?|m[\\\\\\\\'\\\"]*(?:u[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*e|d[\\\\\\\\'\\\"]*i)[\\\\\\\\'\\\"]*r|n[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*o|s[\\\\\\\\'\\\"]*y[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*c|c[\\\\\\\\'\\\"]*p)|b[\\\\\\\\'\\\"]*(?:z[\\\\\\\\'\\\"]*(?:(?:[ef][\\\\\\\\'\\\"]*)?g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p|d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s|m[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e|c[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t)|s[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*(?:c[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t|i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|t[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*r)|u[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n|a[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*h)|m[\\\\\\\\'\\\"]*(?:y[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*q[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*(?:d[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*p(?:[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*w)?|h[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*y|a[\\\\\\\\'\\\"]*d[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n|s[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*w)|l[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*e|a[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*q)|u[\\\\\\\\'\\\"]*(?:n[\\\\\\\\'\\\"]*(?:c[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s|l[\\\\\\\\'\\\"]*z[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*a|a[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*e|r[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*r|s[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*t|z[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*p|x[\\\\\\\\'\\\"]*z)|s[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*(?:(?:a[\\\\\\\\'\\\"]*d|m[\\\\\\\\'\\\"]*o)[\\\\\\\\'\\\"]*d|d[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*l))|x[\\\\\\\\'\\\"]*(?:z(?:[\\\\\\\\'\\\"]*(?:(?:[ef][\\\\\\\\'\\\"]*)?g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p|d[\\\\\\\\'\\\"]*(?:i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|e[\\\\\\\\'\\\"]*c)|c[\\\\\\\\'\\\"]*(?:a[\\\\\\\\'\\\"]*t|m[\\\\\\\\'\\\"]*p)|l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s|m[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e))?|a[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*g[\\\\\\\\'\\\"]*s)|z[\\\\\\\\'\\\"]*(?:(?:(?:[ef][\\\\\\\\'\\\"]*)?g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e|i)[\\\\\\\\'\\\"]*p|c[\\\\\\\\'\\\"]*(?:a[\\\\\\\\'\\\"]*t|m[\\\\\\\\'\\\"]*p)|d[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*s|m[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e|r[\\\\\\\\'\\\"]*u[\\\\\\\\'\\\"]*n|s[\\\\\\\\'\\\"]*h)|f[\\\\\\\\'\\\"]*(?:t[\\\\\\\\'\\\"]*p[\\\\\\\\'\\\"]*(?:s[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*s|w[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*o)|i[\\\\\\\\'\\\"]*l[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*t|e[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*c[\\\\\\\\'\\\"]*h|g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p)|c[\\\\\\\\'\\\"]*(?:o[\\\\\\\\'\\\"]*(?:m[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*d|p[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*c)|u[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*l|s[\\\\\\\\'\\\"]*h|c)|e[\\\\\\\\'\\\"]*(?:g[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p|c[\\\\\\\\'\\\"]*h[\\\\\\\\'\\\"]*o|v[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*l|x[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*c|n[\\\\\\\\'\\\"]*v)|d[\\\\\\\\'\\\"]*(?:m[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*g|a[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*h|i[\\\\\\\\'\\\"]*f[\\\\\\\\'\\\"]*f|o[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*s)|g[\\\\\\\\'\\\"]*(?:z[\\\\\\\\'\\\"]*(?:c[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*t|i[\\\\\\\\'\\\"]*p)|r[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*p|c[\\\\\\\\'\\\"]*c)|j[\\\\\\\\'\\\"]*(?:o[\\\\\\\\'\\\"]*b[\\\\\\\\'\\\"]*s[\\\\\\\\'\\\"]*\\s+[\\\\\\\\'\\\"]*-[\\\\\\\\'\\\"]*x|a[\\\\\\\\'\\\"]*v[\\\\\\\\'\\\"]*a)|w[\\\\\\\\'\\\"]*(?:h[\\\\\\\\'\\\"]*o[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*m[\\\\\\\\'\\\"]*i|g[\\\\\\\\'\\\"]*e[\\\\\\\\'\\\"]*t|3[\\\\\\\\'\\\"]*m)|i[\\\\\\\\'\\\"]*r[\\\\\\\\'\\\"]*b(?:[\\\\\\\\'\\\"]*(?:1(?:[\\\\\\\\'\\\"]*[89])?|2[\\\\\\\\'\\\"]*[012]))?|o[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*n[\\\\\\\\'\\\"]*t[\\\\\\\\'\\\"]*r|h[\\\\\\\\'\\\"]*(?:e[\\\\\\\\'\\\"]*a[\\\\\\\\'\\\"]*d|u[\\\\\\\\'\\\"]*p)|v[\\\\\\\\'\\\"]*i[\\\\\\\\'\\\"]*(?:g[\\\\\\\\'\\\"]*r|p[\\\\\\\\'\\\"]*w)|G[\\\\\\\\'\\\"]*E[\\\\\\\\'\\\"]*T)[\\\\\\\\'\\\"]*(?:\\s|;|\\||&|<|>)"
};

/* ********************************** */

/**
 * [ Unix shell snippets ]
 *
 * Detect some common sequences found in shell commands and scripts.
 *
 * This regex is also triggered by an Apache Struts Remote Code Execution exploit:
 * [ Apache Struts vulnerability CVE-2017-9805 - Exploit tested: https://www.exploit-db.com/exploits/42627 ]
 *
 * This regex is also triggered by an Oracle WebLogic Remote Command Execution exploit:
 * [ Oracle WebLogic vulnerability CVE-2017-10271 - Exploit tested: https://www.exploit-db.com/exploits/43458 ]
 */

static const char *ush_commands[] = {
  "${CDPATH}",
  "${DIRSTACK}",
  "${HOME}",
  "${HOSTNAME}",
  "${IFS}",
  "${OLDPWD}",
  "${OSTYPE}",
  "${PATH}",
  "${PWD}",
  "$CDPATH",
  "$DIRSTACK",
  "$HOME",
  "$HOSTNAME",
  "$IFS",
  "$OLDPWD",
  "$OSTYPE",
  "$PATH",
  "$PWD",
  "bin/bash",
  "bin/cat",
  "bin/csh",
  "bin/dash",
  "bin/du",
  "bin/echo",
  "bin/grep",
  "bin/less",
  "bin/ls",
  "bin/mknod",
  "bin/more",
  "bin/nc",
  "bin/ps",
  "bin/rbash",
  "bin/sh",
  "bin/sleep",
  "bin/su",
  "bin/tcsh",
  "bin/uname",
  "dev/fd/",
  "dev/null",
  "dev/stderr",
  "dev/stdin",
  "dev/stdout",
  "dev/tcp/",
  "dev/udp/",
  "dev/zero",
  "etc/group",
  "etc/master.passwd",
  "etc/passwd",
  "etc/pwd.db",
  "etc/shadow",
  "etc/shells",
  "etc/spwd.db",
  "proc/self/",
  "usr/bin/awk",
  "usr/bin/base64",
  "usr/bin/cat",
  "usr/bin/cc",
  "usr/bin/clang",
  "usr/bin/clang++",
  "usr/bin/curl",
  "usr/bin/diff",
  "usr/bin/env",
  "usr/bin/fetch",
  "usr/bin/file",
  "usr/bin/find",
  "usr/bin/ftp",
  "usr/bin/gawk",
  "usr/bin/gcc",
  "usr/bin/head",
  "usr/bin/hexdump",
  "usr/bin/id",
  "usr/bin/less",
  "usr/bin/ln",
  "usr/bin/mkfifo",
  "usr/bin/more",
  "usr/bin/nc",
  "usr/bin/ncat",
  "usr/bin/nice",
  "usr/bin/nmap",
  "usr/bin/perl",
  "usr/bin/php",
  "usr/bin/php5",
  "usr/bin/php7",
  "usr/bin/php-cgi",
  "usr/bin/printf",
  "usr/bin/psed",
  "usr/bin/python",
  "usr/bin/python2",
  "usr/bin/python3",
  "usr/bin/ruby",
  "usr/bin/sed",
  "usr/bin/socat",
  "usr/bin/tail",
  "usr/bin/tee",
  "usr/bin/telnet",
  "usr/bin/top",
  "usr/bin/uname",
  "usr/bin/wget",
  "usr/bin/who",
  "usr/bin/whoami",
  "usr/bin/xargs",
  "usr/bin/xxd",
  "usr/bin/yes",
  "usr/local/bin/bash",
  "usr/local/bin/curl",
  "usr/local/bin/ncat",
  "usr/local/bin/nmap",
  "usr/local/bin/perl",
  "usr/local/bin/php",
  "usr/local/bin/python",
  "usr/local/bin/python2",
  "usr/local/bin/python3",
  "usr/local/bin/rbash",
  "usr/local/bin/ruby",
  "usr/local/bin/wget"
};

/* ********************************** */

/**
 * [ Windows PowerShell, cmdlets and options ]
 *
 * Detect some common PowerShell commands, cmdlets and options.
 * These commands should be relatively uncommon in normal text, but
 * potentially useful for code injection.
 */

static const char *pwsh_commands[] = {
  "powershell.exe",
  "Add-BitsFile",
  "Add-Computer",
  "Add-Content",
  "Add-History",
  "Add-Member",
  "Add-PSSnapin",
  "Add-Type",
  "Checkpoint-Computer",
  "Clear-Content",
  "Clear-EventLog",
  "Clear-History",
  "Clear-Item",
  "Clear-ItemProperty",
  "Clear-Variable",
  "Compare-Object",
  "Complete-BitsTransfer",
  "Complete-Transaction",
  "Connect-WSMan",
  "ConvertFrom-CSV",
  "ConvertFrom-SecureString",
  "ConvertFrom-StringData",
  "Convert-Path",
  "ConvertTo-CSV",
  "ConvertTo-Html",
  "ConvertTo-SecureString",
  "ConvertTo-XML",
  "Copy-Item",
  "Copy-ItemProperty",
  "Debug-Process",
  "Disable-ComputerRestore",
  "Disable-PSBreakpoint",
  "Disable-PSSessionConfiguration",
  "Disable-WSManCredSSP",
  "Disconnect-WSMan",
  "Enable-ComputerRestore",
  "Enable-PSBreakpoint",
  "Enable-PSRemoting",
  "Enable-PSSessionConfiguration",
  "Enable-WSManCredSSP",
  "Enter-PSSession",
  "Exit-PSSession",
  "Export-Alias",
  "Export-Clixml",
  "Export-Console",
  "Export-Counter",
  "Export-CSV",
  "Export-FormatData",
  "Export-ModuleMember",
  "Export-PSSession",
  "ForEach-Object",
  "Format-Custom",
  "Format-List",
  "Format-Table",
  "Format-Wide",
  "Get-Acl",
  "Get-Alias",
  "Get-AppLockerFileInformation",
  "Get-AppLockerPolicy",
  "Get-AuthenticodeSignature",
  "Get-BitsTransfer",
  "Get-ChildItem",
  "Get-Command",
  "Get-ComputerRestorePoint",
  "Get-Content",
  "Get-Counter",
  "Get-Credential",
  "Get-Culture",
  "Get-Event",
  "Get-EventLog",
  "Get-EventSubscriber",
  "Get-ExecutionPolicy",
  "Get-FormatData",
  "Get-History",
  "Get-Host",
  "Get-HotFix",
  "Get-Item",
  "Get-ItemProperty",
  "Get-Job",
  "Get-Location",
  "Get-Member",
  "Get-Module",
  "Get-PfxCertificate",
  "Get-Process",
  "Get-PSBreakpoint",
  "Get-PSCallStack",
  "Get-PSDrive",
  "Get-PSProvider",
  "Get-PSSession",
  "Get-PSSessionConfiguration",
  "Get-PSSnapin",
  "Get-Random",
  "Get-Service",
  "Get-TraceSource",
  "Get-Transaction",
  "Get-TroubleshootingPack",
  "Get-UICulture",
  "Get-Unique",
  "Get-Variable",
  "Get-WinEvent",
  "Get-WmiObject",
  "Get-WSManCredSSP",
  "Get-WSManInstance",
  "Group-Object",
  "Import-Alias",
  "Import-Clixml",
  "Import-Counter",
  "Import-CSV",
  "Import-LocalizedData",
  "Import-Module",
  "Import-PSSession",
  "Invoke-Command",
  "Invoke-Expression",
  "Invoke-History",
  "Invoke-Item",
  "Invoke-TroubleshootingPack",
  "Invoke-WmiMethod",
  "Invoke-WSManAction",
  "Join-Path",
  "Limit-EventLog",
  "Measure-Command",
  "Measure-Object",
  "Move-Item",
  "Move-ItemProperty",
  "New-Alias",
  "New-AppLockerPolicy",
  "New-Event",
  "New-EventLog",
  "New-Item",
  "New-ItemProperty",
  "New-Module",
  "New-ModuleManifest",
  "New-Object",
  "New-PSDrive",
  "New-PSSession",
  "New-PSSessionOption",
  "New-Service",
  "New-TimeSpan",
  "New-Variable",
  "New-WebServiceProxy",
  "New-WSManInstance",
  "New-WSManSessionOption",
  "Out-Default",
  "Out-File",
  "Out-GridView",
  "Out-Host",
  "Out-Null",
  "Out-Printer",
  "Out-String",
  "Pop-Location",
  "Push-Location",
  "Read-Host",
  "Receive-Job",
  "Register-EngineEvent",
  "Register-ObjectEvent",
  "Register-PSSessionConfiguration",
  "Register-WmiEvent",
  "Remove-BitsTransfer",
  "Remove-Computer",
  "Remove-Event",
  "Remove-EventLog",
  "Remove-Item",
  "Remove-ItemProperty",
  "Remove-Job",
  "Remove-Module",
  "Remove-PSBreakpoint",
  "Remove-PSDrive",
  "Remove-PSSession",
  "Remove-PSSnapin",
  "Remove-Variable",
  "Remove-WmiObject",
  "Remove-WSManInstance",
  "Rename-Item",
  "Rename-ItemProperty",
  "Reset-ComputerMachinePassword",
  "Resolve-Path",
  "Restart-Computer",
  "Restart-Service",
  "Restore-Computer",
  "Resume-BitsTransfer",
  "Resume-Service",
  "Select-Object",
  "Select-String",
  "Select-XML",
  "Send-MailMessage",
  "Set-Acl",
  "Set-Alias",
  "Set-AppLockerPolicy",
  "Set-AuthenticodeSignature",
  "Set-BitsTransfer",
  "Set-Content",
  "Set-Date",
  "Set-ExecutionPolicy",
  "Set-Item",
  "Set-ItemProperty",
  "Set-Location",
  "Set-PSBreakpoint",
  "Set-PSDebug",
  "Set-PSSessionConfiguration",
  "Set-Service",
  "Set-StrictMode",
  "Set-TraceSource",
  "Set-Variable",
  "Set-WmiInstance",
  "Set-WSManInstance",
  "Set-WSManQuickConfig",
  "Show-EventLog",
  "Sort-Object",
  "Split-Path",
  "Start-BitsTransfer",
  "Start-Job",
  "Start-Process",
  "Start-Service",
  "Start-Sleep",
  "Start-Transaction",
  "Start-Transcript",
  "Stop-Computer",
  "Stop-Job",
  "Stop-Process",
  "Stop-Service",
  "Stop-Transcript",
  "Suspend-BitsTransfer",
  "Suspend-Service",
  "Tee-Object",
  "Test-AppLockerPolicy",
  "Test-ComputerSecureChannel",
  "Test-Connection",
  "Test-ModuleManifest",
  "Test-Path",
  "Test-WSMan",
  "Trace-Command",
  "Undo-Transaction",
  "Unregister-Event",
  "Unregister-PSSessionConfiguration",
  "Update-FormatData",
  "Update-List",
  "Update-TypeData",
  "Use-Transaction",
  "Wait-Event",
  "Wait-Job",
  "Wait-Process",
  "Where-Object",
  "Write-Debug",
  "Write-Error",
  "Write-EventLog",
  "Write-Host",
  "Write-Output",
  "Write-Progress",
  "Write-Verbose",
  "Write-Warning",
  "-EncodedCommand",
  "-ExecutionPolicy",
  "-PSConsoleFile"
};

#endif //HAVE_PCRE2