#!/bin/bash

set -eo pipefail

. ./VERSION

[ -n "$NETIFY_MINVER" -a -n "$NETIFY_MINVER_OVERRIDE" ] && \
    MINVER="$NETIFY_MINVER_OVERRIDE"
[ -n "$NETIFY_MINVER_RELEASE" -a -n "$NETIFY_MINVER_RELEASE_OVERRIDE" ] && \
    MINVER_REL="$NETIFY_MINVER_RELEASE_OVERRIDE"
[ -n "$NETIFY_PLM_MINVER" -a -n "$NETIFY_PLM_MINVER_OVERRIDE" ] && \
    PLM_MINVER="$NETIFY_PLM_MINVER_OVERRIDE"
[ -n "$NETIFY_PLM_MINVER_RELEASE" -a -n "$NETIFY_PLM_MINVER_RELEASE_OVERRIDE" ] \
    && PLM_MINVER_REL="$NETIFY_PLM_MINVER_RELEASE_OVERRIDE"

get_version() {
    local VER="$2"
    local VER_NEW
    read -e -i "$VER" -p "New $1 version: " VER_NEW

    if [ "$VER_NEW" == "$VER" ]; then
        echo "unchanged"
        return
    fi

    if [[ ! "$VER_NEW" =~ ^([0-9]+)\.([0-9]+)\.([0-9]+)$ ]]; then
        echo "invalid"
        return
    fi

    echo "$VER_NEW"
}

get_version_hex() {
    local VER="$1"
    local REL="$2"
    local DIGIT

    echo -n "0x"
    for DIGIT in $(echo $VER | sed -e 's/\./ /g'); do
        printf "%02x" $DIGIT
    done
    printf "%02x\n" $REL
}

get_release() {
    local REL="$2"
    local REL_NEW
    read -e -i "$REL" -p "New $1 release: " REL_NEW

    if [ "$REL_NEW" == "$REL" ]; then
        echo "unchanged"
        return
    fi

    if [[ ! "$REL_NEW" =~ ^([0-9]+)$ ]]; then
        echo "invalid"
        return
    fi

    echo "$REL_NEW"
}

VER="$(get_version "project" "$PROJECT_VERSION")"
case "$VER" in
  unchanged)
    echo "Version unchanged..."
    VER="$PROJECT_VERSION"
    ;;
  invalid)
    echo "Version invalid; aborting..."
    exit 2
    ;;
  *)
    PROJECT_RELEASE=1
    ;;
esac

REL="$(get_release "project" "$PROJECT_RELEASE")"
case "$REL" in
  unchanged)
    echo "Release unchanged..."
    REL="$PROJECT_RELEASE"
    ;;
  invalid)
    echo "Release invalid; aborting..."
    exit 2
    ;;
esac

if [ -n "$NETIFY_MINVER" -a -z "$NETIFY_MINVER_OVERRIDE" ]; then
    MINVER="$(get_version "agent" "$NETIFY_MINVER")"
    case "$MINVER" in
      unchanged)
        echo "Version unchanged..."
        MINVER="$NETIFY_MINVER"
        ;;
      invalid)
        echo "Version invalid; aborting..."
        exit 2
        ;;
      *)
        if [ -n "$NETIFY_MINVER_RELEASE" ]; then
            NETIFY_MINVER_RELEASE=1
        fi
        ;;
    esac
fi

if [ -n "$NETIFY_MINVER_RELEASE" -a -z "$NETIFY_MINVER_RELEASE_OVERRIDE" ]; then
    MINVER_REL="$(get_release "agent" "$NETIFY_MINVER_RELEASE")"
    case "$MINVER_REL" in
      unchanged)
        echo "Release unchanged..."
        MINVER_REL="$NETIFY_MINVER_RELEASE"
        ;;
      invalid)
        echo "Release invalid; aborting..."
        exit 2
        ;;
    esac
fi

if [ -n "$NETIFY_PLM_MINVER" -a -z "$NETIFY_PLM_MINVER_OVERRIDE" ]; then
    PLM_MINVER="$(get_version "PLM" "$NETIFY_PLM_MINVER")"
    case "$PLM_MINVER" in
      unchanged)
        echo "Version unchanged..."
        PLM_MINVER="$NETIFY_PLM_MINVER"
        ;;
      invalid)
        echo "Version invalid; aborting..."
        exit 2
        ;;
      *)
        if [ "$NETIFY_PLM_MINVER_RELEASE" ]; then
            NETIFY_PLM_MINVER_RELEASE=1
        fi
        ;;
    esac
fi

if [ -n "$NETIFY_PLM_MINVER_RELEASE" -a -z "$NETIFY_PLM_MINVER_RELEASE_OVERRIDE" ]; then
    PLM_MINVER_REL="$(get_release "PLM" "$NETIFY_PLM_MINVER_RELEASE")"
    case "$PLM_MINVER_REL" in
      unchanged)
        echo "Release unchanged..."
        PLM_MINVER_REL="$NETIFY_PLM_MINVER_RELEASE"
        ;;
      invalid)
        echo "Release invalid; aborting..."
        exit 2
        ;;
    esac
fi

printf "PROJECT_VERSION=\"%s\"\n" "$VER" > VERSION
printf "PROJECT_VERSION_HEX=\"%s\"\n" $(get_version_hex "$VER" "$REL") >> VERSION
printf "PROJECT_RELEASE=\"%s\"\n" "$REL" >> VERSION

if [ -n "$MINVER" ]; then
    printf "NETIFY_MINVER=\"%s\"\n" "$MINVER" >> VERSION
    if [ -f .gitlab-ci.yml ]; then
        sed -i -E "s/NETIFY_MINVER:.*/NETIFY_MINVER: \"$MINVER\"/" .gitlab-ci.yml
    fi
fi
if [ -n "$MINVER_REL" ]; then
    printf "NETIFY_MINVER_RELEASE=\"%s\"\n" "$MINVER_REL" >> VERSION
    if [ -f .gitlab-ci.yml ]; then
        sed -i -E "s/NETIFY_MINVER_RELEASE:.*/NETIFY_MINVER_RELEASE: \"$MINVER_REL\"/" .gitlab-ci.yml
    fi
fi
if [ -n "$PLM_MINVER" ]; then
    printf "NETIFY_PLM_MINVER=\"%s\"\n" "$PLM_MINVER" >> VERSION
    if [ -f .gitlab-ci.yml ]; then
        sed -i -E "s/NETIFY_PLM_MINVER:.*/NETIFY_PLM_MINVER: \"$PLM_MINVER\"/" .gitlab-ci.yml
    fi
fi
if [ -n "$PLM_MINVER_REL" ]; then
    printf "NETIFY_PLM_MINVER_RELEASE=\"%s\"\n" "$PLM_MINVER_REL" >> VERSION
    if [ -f .gitlab-ci.yml ]; then
        sed -i -E "s/NETIFY_PLM_MINVER_RELEASE:.*/NETIFY_PLM_MINVER_RELEASE: \"$PLM_MINVER_REL\"/" .gitlab-ci.yml
    fi
fi

exit 0
